
// __________________________________________________________
//
//                         win_pdbx.c
//            SBS Program Database Exploder V1.00
//                06-25-2006 Sven B. Schreiber
//                       sbs@orgon.com
// __________________________________________________________

#include "win_pdbx.h"

// =================================================================
// DISCLAIMER
// =================================================================

/*

This software is provided "as is" and any express or implied
warranties, including, but not limited to, the implied warranties of
merchantability and fitness for a particular purpose are disclaimed.
In no event shall the author Sven B. Schreiber be liable for any
direct, indirect, incidental, special, exemplary, or consequential
damages (including, but not limited to, procurement of substitute
goods or services; loss of use, data, or profits; or business
interruption) however caused and on any theory of liability,
whether in contract, strict liability, or tort (including negligence
or otherwise) arising in any way out of the use of this software,
even if advised of the possibility of such damage.

*/

// =================================================================
// REVISION HISTORY
// =================================================================

/*

06-25-2006 V1.00 Original version (SBS).

*/

// =================================================================
// GLOBAL STRINGS
// =================================================================

TBYTE atArguments [] = T("[/types] <file list>");

TBYTE atInfo [] =
    T("\r\n")
    T("This program extracts and reassembles streams packed in\r\n")
    T("Microsoft Program Database (PDB) files. It supports the\r\n")
    T("PDB format versions 2.00 and 7.00. The stream contents\r\n")
    T("are written to individual binary files named after the\r\n")
    T("source files by appending 3-digit stream index numbers.\r\n")
    T("\r\n")
    T("The /types option results in a structured dump of the\r\n")
    T("type info stream #002 (PDB_STREAM_TPI) to be written\r\n")
    T("to the console.\r\n");

// =================================================================
// PDB STREAM MANAGEMENT
// =================================================================

BOOL WINAPI SaveStreams (PTBYTE ptSource)
    {
    PTBYTE      ptTarget;
    PPDB_DATA   ppd;
    PPDB_STREAM pps;
    DWORD       i, j;
    BOOL        fInvalid;
    BOOL        fOk = FALSE;

    if ((ppd = pdbDataOpen (ptSource, &fInvalid)) != NULL)
        {
        fOk = TRUE;

        _printf (T("\r\n")
                 T("Path:     \"%ls\"\r\n")
                 T("Version:  %lu.%02lu\r\n")
                 T("Streams:  %lu\r\n")
                 T("Unused:   "),
                 ppd->awPath,
                 ppd->dVersion / 100, ppd->dVersion % 100,
                 ppd->dStreams);

        for (i = j = 0; i < ppd->dStreams; i++)
            {
            pps = ppd->aStreams + i;

            if (pps->pData != NULL)
                {
                ptTarget = _bprintf (NULL, T("%s.%03lu"),
                                           ptSource, i);

                if (!_fsave (ptTarget, pps->pData, pps->dData))
                    {
                    fOk = FALSE;
                    }
                _mfree (ptTarget);
                }
            else
                {
                if (pps->fUnused)
                    {
                    _printf (T("%s%lu"),
                             (j++ ? T(", ") : NULL), i);
                    }
                }
            }
        _printf (T("%s\r\n"),
                 (j ? NULL : T("none")));

        pdbDataClose (ppd);
        }
    else
        {
        if (fInvalid)
            {
            _printf (T("\r\n\"%s\" is not a valid PDB file.\r\n"),
                     _ffile (ptSource));
            }
        else
            {
            _printf (T("\r\nError loading file \"%s\".\r\n"),
                     _ffile (ptSource));
            }
        }
    return fOk;
    }

// =================================================================
// TYPE INFO BROWSER
// =================================================================

PBYTE WINAPI RecordValue (PBYTE  pbData,
                          PDWORD pdValue)
    {
    WORD  wValue;
    DWORD dValue = -1;
    PBYTE pbText = NULL;

    if (pbData != NULL)
        {
        if ((wValue = *(PWORD) pbData) < LF_NUMERIC)
            {
            dValue = wValue;
            pbText = pbData + WORD_;
            }
        else
            {
            switch (wValue)
                {
                case LF_CHAR:
                    {
                    dValue = (LONG) (*(PCHAR  ) (pbData + WORD_));
                    pbText = pbData + WORD_ + CHAR_;
                    break;
                    }
                case LF_SHORT:
                    {
                    dValue = (LONG) (*(PSHORT ) (pbData + WORD_));
                    pbText = pbData + WORD_ + SHORT_;
                    break;
                    }
                case LF_USHORT:
                    {
                    dValue = (LONG) (*(PUSHORT) (pbData + WORD_));
                    pbText = pbData + WORD_ + USHORT_;
                    break;
                    }
                case LF_LONG:
                    {
                    dValue = (LONG) (*(PLONG  ) (pbData + WORD_));
                    pbText = pbData + WORD_ + LONG_;
                    break;
                    }
                case LF_ULONG:
                    {
                    dValue = (LONG) (*(PULONG ) (pbData + WORD_));
                    pbText = pbData + WORD_ + ULONG_;
                    break;
                    }
                }
            }
        }
    if (pdValue != NULL) *pdValue = dValue;
    return pbText;
    }

// -----------------------------------------------------------------

PBYTE WINAPI MethodValue (CV_fldattr_t attr,
                          PDWORD       pdData,
                          PDWORD       pdValue)
    {
    DWORD dValue = -1;
    PBYTE pbText = NULL;

    if (pdData != NULL)
        {
        switch (attr.mprop)
            {
            case CV_MTintro:
            case CV_MTpureintro:
                {
                dValue = *pdData;
                pbText = (PBYTE) (pdData + 1);
                break;
                }
            default:
                {
                pbText = (PBYTE) pdData;
                break;
                }
            }
        }
    if (pdValue != NULL) *pdValue = dValue;
    return pbText;
    }

// -----------------------------------------------------------------

VOID WINAPI DisplayArray (PlfArray pla,
                          DWORD    dBase,
                          DWORD    dSize)
    {
    DWORD dBytes;
    PBYTE pbName = RecordValue (pla->data, &dBytes);

    _printf (T(" array   %08lX %08lX %08lX\r\n"),
             pla->elemtype, pla->idxtype, dBytes);
    return;
    }

// -----------------------------------------------------------------

VOID WINAPI DisplayBitfield (PlfBitfield plb,
                             DWORD       dBase,
                             DWORD       dSize)
    {
    _printf (T(" bitfield (%08lX) %02lX : %02lX\r\n"),
             plb->type, plb->position, plb->length);
    return;
    }

// -----------------------------------------------------------------

VOID WINAPI DisplayClass (PlfClass plc,
                          DWORD    dBase,
                          DWORD    dSize)
    {
    DWORD dBytes;
    PBYTE pbName = RecordValue (plc->data, &dBytes);

    _printf (T(" class   %08lX %08lX %04hX %08lX %08lX [%hs]\r\n"),
             plc->field, dBytes, plc->count,
             plc->derived, plc->vshape, pbName);
    return;
    }

// -----------------------------------------------------------------

VOID WINAPI DisplayStructure (PlfStructure pls,
                              DWORD        dBase,
                              DWORD        dSize)
    {
    DWORD dBytes;
    PBYTE pbName = RecordValue (pls->data, &dBytes);

    _printf (T(" struct  %08lX %08lX %04hX %08lX %08lX [%hs]\r\n"),
             pls->field, dBytes, pls->count,
             pls->derived, pls->vshape, pbName);
    return;
    }

// -----------------------------------------------------------------

VOID WINAPI DisplayUnion (PlfUnion plu,
                          DWORD    dBase,
                          DWORD    dSize)
    {
    DWORD dBytes;
    PBYTE pbName = RecordValue (plu->data, &dBytes);

    _printf (T(" union   %08lX %08lX %04hX [%hs]\r\n"),
             plu->field, dBytes, plu->count, pbName);
    return;
    }

// -----------------------------------------------------------------

VOID WINAPI DisplayEnum (PlfEnum ple,
                         DWORD   dBase,
                         DWORD   dSize)
    {
    _printf (T(" enum    %08lX %08lX %04hX [%hs]\r\n"),
             ple->field, ple->utype, ple->count, ple->Name);
    return;
    }

// -----------------------------------------------------------------

VOID WINAPI DisplayPointer (PlfPointer plp,
                            DWORD      dBase,
                            DWORD      dSize)
    {
    _printf (T(" pointer %08lX\r\n"),
             plp->utype);
    return;
    }

// -----------------------------------------------------------------

VOID WINAPI DisplayProc (PlfProc plp,
                         DWORD   dBase,
                         DWORD   dSize)
    {
    _printf (T(" proc    %08lX %08lX %04lX %02lX\r\n"),
             plp->rvtype, plp->arglist, plp->parmcount,
             plp->calltype);
    return;
    }

// -----------------------------------------------------------------

VOID WINAPI DisplayMFunc (PlfMFunc plmf,
                          DWORD    dBase,
                          DWORD    dSize)
    {
    _printf (T(" mfunc   %08lX %08lX %04lX %02lX %08lX %08lX %08lX\r\n"),
             plmf->rvtype, plmf->arglist, plmf->parmcount,
             plmf->calltype, plmf->classtype, plmf->thistype,
             plmf->thisadjust);
    return;
    }

// -----------------------------------------------------------------

VOID WINAPI DisplayArgList (PlfArgList plal,
                            DWORD      dBase,
                            DWORD      dSize)
    {
    DWORD i;

    _printf (T(" arglist %08lX"),
             plal->count);

    for (i = 0; i < plal->count; i++)
        {
        _printf (T(" %08lX"), plal->arg [i]);
        }
    _printf (T("\r\n"));
    return;
    }

// -----------------------------------------------------------------

VOID WINAPI DisplayVTShape (PlfVTShape plvts,
                            DWORD      dBase,
                            DWORD      dSize)
    {
    DWORD i;
    BYTE  b;

    _printf (T(" vtshape %08lX"),
             plvts->count);

    for (i = 0; i < plvts->count; i++)
        {
        b = plvts->desc [i/2];
        _printf (T(" %lX"), (i & 1 ? b & 0xF : b >> 4));
        }
    _printf (T("\r\n"));
    return;
    }

// -----------------------------------------------------------------

VOID WINAPI DisplayFieldList (PlfFieldList plfl,
                              DWORD        dBase,
                              DWORD        dSize)
    {
    PlfSubRecord plsr;
    DWORD        dValue, dOffset, i, n;
    PBYTE        pbName, pbNext;

    _printf (T("\r\n"));

    for (i = 0; dSize - i >= lfFieldList_; i += n)
        {
        plsr = SKIP (&plfl->SubRecord, i);

        _printf (T("        %04hX"), plsr->leaf);

        switch (plsr->leaf)
            {
            case LF_ENUMERATE:
                {
                pbName = RecordValue (plsr->Enumerate.value,
                                      &dValue);

                _printf (T(" const  %08lX [%hs]\r\n"),
                         dValue, pbName);

                n = ((DWORD_PTR) pbName - (DWORD_PTR) plsr) +
                    _tsizeA (pbName) + 1;
                break;
                }
            case LF_MEMBER:
                {
                pbName = RecordValue (plsr->Member.offset,
                                      &dOffset);

                _printf (T(" field  %08lX (%08lX) [%hs]\r\n"),
                         dOffset, plsr->Member.index, pbName);

                n = ((DWORD_PTR) pbName - (DWORD_PTR) plsr) +
                    _tsizeA (pbName) + 1;
                break;
                }
            case LF_BCLASS:
                {
                pbNext = RecordValue (plsr->BClass.offset,
                                      &dOffset);

                _printf (T(" bclass %08lX (%08lX)\r\n"),
                         dOffset, plsr->BClass.index);

                n = (DWORD_PTR) pbNext - (DWORD_PTR) plsr;
                break;
                }
            case LF_VFUNCTAB:
                {
                _printf (T(" vfunction table (%08lX)\r\n"),
                         plsr->VFuncTab.type);

                n = lfVFuncTab_;
                break;
                }
            case LF_ONEMETHOD:
                {
                pbName = MethodValue (plsr->OneMethod.attr,
                                      plsr->OneMethod.vbaseoff,
                                      &dValue);

                _printf (T(" single %08lX (%08lX) [%hs]\r\n"),
                         dValue, plsr->OneMethod.index, pbName);

                n = ((DWORD_PTR) pbName - (DWORD_PTR) plsr) +
                    _tsizeA (pbName) + 1;
                break;
                }
            case LF_METHOD:
                {
                _printf (T(" method %08lX (%08lX) [%hs]\r\n"),
                         plsr->Method.count,
                         plsr->Method.mList,
                         plsr->Method.Name);

                n = ((DWORD_PTR) plsr->Method.Name - (DWORD_PTR) plsr) +
                    _tsizeA (plsr->Method.Name) + 1;
                break;
                }
            case LF_NESTTYPE:
                {
                _printf (T(" nested typedef  (%08lX) [%hs]\r\n"),
                         plsr->NestType.index, plsr->NestType.Name);

                n = ((DWORD_PTR) plsr->NestType.Name - (DWORD_PTR) plsr) +
                    _tsizeA (plsr->NestType.Name) + 1;
                break;
                }
            default:
                {
                _printf (T(" member ###\r\n"));
                n = 0;
                break;
                }
            }
        if (!(n = (n + (DWORD_ - 1)) & (0 - DWORD_))) break;
        }
    return;
    }

// -----------------------------------------------------------------

VOID WINAPI DisplayRecord (PlfRecord plr,
                           DWORD     dBase,
                           DWORD     dSize)
    {
    _printf (T(" ???\r\n"));
    return;
    }

// -----------------------------------------------------------------

BOOL WINAPI DisplayTypes (PTBYTE ptSource)
    {
    PPDB_DATA ppd;
    PHDR      pHdr;
    PlfRecord plr;
    PVOID     pData;
    DWORD     dData, dTypes, dBase, dSize, i;
    BOOL      fInvalid;
    BOOL      fOk = FALSE;

    if ((ppd = pdbDataOpen (ptSource, &fInvalid)) != NULL)
        {
        fOk = TRUE;

        if (PDB_STREAM_TPI < ppd->dStreams)
            {
            pHdr  = ppd->aStreams [PDB_STREAM_TPI].pData;
            dData = ppd->aStreams [PDB_STREAM_TPI].dData;

            if ((dData >= HDR_) &&
                (dData >= pHdr->cbHdr + pHdr->cbGprec) &&
                (pHdr->tiMac > pHdr->tiMin))
                {
                dTypes = pHdr->tiMac - pHdr->tiMin;
                dBase  = pHdr->cbHdr;
                pData  = SKIP (pHdr, dBase);

                _printf (T("\r\n")
                         T("TPI Version:  %lu\r\n")
                         T("Index range:  %lX..%lX\r\n")
                         T("Type count:   %lu\r\n"),
                         pHdr->vers,
                         pHdr->tiMin, pHdr->tiMac-1,
                         dTypes);

                _printf (T("\r\n")
                         T("HDR.vers                      = %lu\r\n")
                         T("HDR.cbHdr                     = 0x%08lX\r\n")
                         T("HDR.tiMin                     = 0x%08lX\r\n")
                         T("HDR.tiMac                     = 0x%08lX\r\n")
                         T("HDR.cbGprec                   = 0x%08lX\r\n")
                         T("HDR.tpihash.sn                = 0x%04hX\r\n")
                         T("HDR.tpihash.snPad             = 0x%04hX\r\n")
                         T("HDR.tpihash.cbHashKey         = 0x%08lX\r\n")
                         T("HDR.tpihash.cHashBuckets      = 0x%08lX\r\n")
                         T("HDR.tpihash.offcbHashVals.off = 0x%08lX\r\n")
                         T("HDR.tpihash.offcbHashVals.cb  = 0x%08lX\r\n")
                         T("HDR.tpihash.offcbTiOff.off    = 0x%08lX\r\n")
                         T("HDR.tpihash.offcbTiOff.cb     = 0x%08lX\r\n")
                         T("HDR.tpihash.offcbHashAdj.off  = 0x%08lX\r\n")
                         T("HDR.tpihash.offcbHashAdj.cb   = 0x%08lX\r\n"),
                         pHdr->vers,
                         pHdr->cbHdr,
                         pHdr->tiMin,
                         pHdr->tiMac,
                         pHdr->cbGprec,
                         pHdr->tpihash.sn,
                         pHdr->tpihash.snPad,
                         pHdr->tpihash.cbHashKey,
                         pHdr->tpihash.cHashBuckets,
                         pHdr->tpihash.offcbHashVals.off,
                         pHdr->tpihash.offcbHashVals.cb,
                         pHdr->tpihash.offcbTiOff.off,
                         pHdr->tpihash.offcbTiOff.cb,
                         pHdr->tpihash.offcbHashAdj.off,
                         pHdr->tpihash.offcbHashAdj.cb);

                _printf (T("\r\n"));

                for (i = 0; i < dTypes; i++)
                    {
                    dSize  = *(PWORD) pData;
                    dBase +=  WORD_;
                    plr    =  SKIP (pHdr, dBase);

                    _printf (T("%6lX: %04hX %08lX"),
                             pHdr->tiMin+i,
                             plr->leaf,
                             dBase - WORD_);

                    switch (plr->leaf)
                        {
                        case LF_ARRAY:
                            {
                            DisplayArray (&plr->Array,
                                          dBase, dSize);
                            break;
                            }
                        case LF_BITFIELD:
                            {
                            DisplayBitfield (&plr->Bitfield,
                                             dBase, dSize);
                            break;
                            }
                        case LF_CLASS:
                            {
                            DisplayClass (&plr->Class,
                                          dBase, dSize);
                            break;
                            }
                        case LF_STRUCTURE:
                            {
                            DisplayStructure (&plr->Structure,
                                              dBase, dSize);
                            break;
                            }
                        case LF_UNION:
                            {
                            DisplayUnion (&plr->Union,
                                          dBase, dSize);
                            break;
                            }
                        case LF_ENUM:
                            {
                            DisplayEnum (&plr->Enum,
                                         dBase, dSize);
                            break;
                            }
                        case LF_POINTER:
                            {
                            DisplayPointer (&plr->Pointer,
                                            dBase, dSize);
                            break;
                            }
                        case LF_PROCEDURE:
                            {
                            DisplayProc (&plr->Proc,
                                         dBase, dSize);
                            break;
                            }
                        case LF_MFUNCTION:
                            {
                            DisplayMFunc (&plr->MFunc,
                                          dBase, dSize);
                            break;
                            }
                        case LF_ARGLIST:
                            {
                            DisplayArgList (&plr->ArgList,
                                            dBase, dSize);
                            break;
                            }
                        case LF_VTSHAPE:
                            {
                            DisplayVTShape (&plr->VTShape,
                                            dBase, dSize);
                            break;
                            }
                        case LF_FIELDLIST:
                            {
                            DisplayFieldList (&plr->FieldList,
                                              dBase, dSize);
                            break;
                            }
                        default:
                            {
                            DisplayRecord (plr,
                                           dBase, dSize);
                            break;
                            }
                        }
                    dBase += dSize;
                    pData  = SKIP (pHdr, dBase);
                    }
                _printf (T("\r\n")
                         T("Offset:  %08lX\r\n"),
                         dBase);
                }
            }
        pdbDataClose (ppd);
        }
    else
        {
        if (fInvalid)
            {
            _printf (T("\r\n\"%s\" is not a valid PDB file.\r\n"),
                     _ffile (ptSource));
            }
        else
            {
            _printf (T("\r\nError loading file \"%s\".\r\n"),
                     _ffile (ptSource));
            }
        }
    return fOk;
    }


// =================================================================
// MAIN PROGRAM
// =================================================================

DWORD Main (DWORD argc, PTBYTE *argv, PTBYTE *argp)
    {
    DWORD dMemoryNow, dMemoryMax, dStart, i;
    BOOL  fTypes;

    _printf (atAbout);

    if (argc < 2)
        {
        _printf (atUsage, atArguments);
        _printf (atInfo);
        }
    else
        {
        dStart = 1;
        fTypes = FALSE;

        if (!_tcompi (argv [dStart], T("/types")))
            {
            dStart++;
            fTypes = TRUE;
            }
        for (i = dStart; i < argc; i++)
            {
            SaveStreams (argv [i]);

            if (fTypes) DisplayTypes (argv [i]);
            }
        _mstatus (&dMemoryNow, &dMemoryMax);

        _printf (T("\r\n")
                 T("RTL memory now:  %lu\r\n")
                 T("RTL memory max:  %lu\r\n"),
                 dMemoryNow, dMemoryMax);
        }
    return 0;
    }

// =================================================================
// END OF PROGRAM
// =================================================================
